<?php


if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly...
}

/**
 * @class PRK_Woocommerce_Order_Tracking
 */

use Automattic\WooCommerce\Internal\DataStores\Orders\CustomOrdersTableController;

class PRK_Woocommerce_Order_Tracking
{

    /**
     * @var object $instance
     */
    private static $instance;

    /**
     * Constructor.
     */
    private function __construct()
    {
        $this->init_text_domain();

        add_action('init', array($this, 'init_hooks'), 0);

    }

    /**
     * Initializes text domain.
     */
    private function init_text_domain()
    {
        load_plugin_textdomain('woocommerce-order-tracking', false, 'woocommerce-order-tracking/languages');
    }

    /**
     * Initializes singleton instance.
     *
     * @return object self::$instance
     */
    public static function get_instance()
    {
        if (is_null(self::$instance)) {
            self::$instance = new self;
        }

        return self::$instance;
    }

    /**
     * Cloning is forbidden.
     */
    public function __clone()
    {
        _doing_it_wrong(__FUNCTION__, __('Cheatin&#8217; huh?', 'woocommerce-order-tracking'), '0.1');
    }

    /**
     * Unserializing instances of this class is forbidden.
     */
    public function __wakeup()
    {
        _doing_it_wrong(__FUNCTION__, __('Cheatin&#8217; huh?', 'woocommerce-order-tracking'), '0.1');
    }


    /**
     * Initialization hooks.
     */
    public function init_hooks()
    {
        if (is_admin()) {

            add_action('admin_enqueue_scripts', array($this, 'registerAdminAssets'));

            // Order page hooks...
            add_action('admin_notices', array($this, 'admin_notice'));
            add_action('add_meta_boxes', array($this, 'add_meta_box'));

            add_action('save_post', array($this, 'save_meta_box'), 10, 3);

            add_filter('is_protected_meta', array($this, 'protect_meta'), 10, 2);

        } else {
            // Frontend hooks...
            add_action('woocommerce_view_order', array($this, 'information_display'), 5, 1);
            add_action('woocommerce_email_order_meta', array($this, 'information_display'), 10, 1);
        }
    }

    function registerAdminAssets()
    {
        $url = get_template_directory_uri() . '/inc/PRKSMSApp/includes/templates/assets/';

        wp_enqueue_style('custom_prk_styles', $url . 'css/customprkstyles.admin.css');
        wp_enqueue_script('custom_prk_script', $url . 'js/customprkscript.admin.js', array('jquery'));

    }

    /**
     * Adds tracking meta box.
     *
     * Hooked into `add_meta_boxes` action hook.
     */
    public function add_meta_box()
    {
        $screen_id = $this->is_hpos_enabled() ? wc_get_page_screen_id('shop-order') : 'shop_order';

        add_meta_box(
            'prk_woocot',
            __('اطلاعات پستی مرسوله', 'woocommerce-order-tracking'),
            array($this, 'meta_box'),
            $screen_id, // استفاده از screen ID سازگار با HPOS
            'side',
            'high'
        );
    }

    public function is_hpos_enabled()
    {
        return class_exists(CustomOrdersTableController::class) &&
            wc_get_container()->get(CustomOrdersTableController::class)->custom_orders_table_usage_is_enabled();
    }

    /**
     * Meta box template.
     */
    public function meta_box()
    {
        $shippers = $this->get_shippers();
        $order = wc_get_order(get_the_ID());

        if ($order) {
            $prk_woocot_shipper = $order->get_meta('prk_woocot_shipper', true);
            $prk_woocot_number = $order->get_meta('prk_woocot_number', true);
            $prk_woocot_transport_type = $order->get_meta('prk_woocot_transport_type', true);
            $prk_woocot_postage_date = $order->get_meta('prk_woocot_postage_date', true);
            $prk_woocot_order_picked_up = $order->get_meta('prk_woocot_order_picked_up', true);
        }

        $prk_woocot_order_picked_up_checked = $prk_woocot_order_picked_up == 'yes' ? 'checked' : '';
        echo '
		<section class="buttons-wrapper">

		<label class="prk_woocot-order-picked-up-label" >سفارش تحویل پست داده شد؟</label>
		<label class="toggler-wrapper style-11">

		<input name="prk_woocot_order_picked_up" type="checkbox" ' . $prk_woocot_order_picked_up_checked . '>
		<div class="toggler-slider">
		  <div class="toggler-knob"></div>
		</div>
	  </label>

		</section>
		';
        echo '<div class="prk_woocot_wrapper" style="display:' . ($prk_woocot_order_picked_up == "yes" ? "block" : "none") . '"><p class="description">' . esc_html__('پس از پر کردن فرم،سفارش را بروزرسانی کنید.', 'woocommerce-order-tracking') . '</p>';
        echo '<p><label for="prk_woocot_shipper">' . esc_html__('انتخاب فرم رهگیری پست :', 'woocommerce-order-tracking') . '</label /><br />';
        echo '<select style="width: 100%;font-weight: 500;background-color: #dce5fb;" id="prk_woocot_shipper" name="prk_woocot_shipper">';
        echo '<option value="">' . esc_html__('هیچکدام', 'woocommerce-order-tracking') . '</option>';

        foreach ($shippers as $key => $value) {
            if (!is_int($key) && !empty($key)) {
                $selected = ($prk_woocot_shipper === $key) ? 'selected ' : '';

                echo '<option ' . $selected . 'value="' . $key . '">' . esc_html($value['name']) . '</option>';
            }
        }

        echo '</select></p>';
        echo '<p><label for="prk_woocot_number">' . esc_html__('درج کد رهگیری :', 'woocommerce-order-tracking') . '</label>';
        echo '<input style="width: 100%;background: #dce5fb;" type="text" id="prk_woocot_number" name="prk_woocot_number" value="' . esc_attr($prk_woocot_number) . '" /></p>';
        echo '<p><label for="prk_woocot_transport_type">' . esc_html__('نوع سیستم حمل و نقل :', 'woocommerce-order-tracking') . '</label>';
        echo '<input style="width: 100%;background: #dce5fb;" type="text" id="prk_woocot_transport_type" name="prk_woocot_transport_type" value="' . esc_attr($prk_woocot_transport_type) . '" /></p>';
        echo '<p><label for="prk_woocot_postage_date">' . esc_html__('تاریخ ارسال :', 'woocommerce-order-tracking') . '</label>';
        echo '<input style="width: 100%;background: #dce5fb;" type="text" id="prk_woocot_postage_date" name="prk_woocot_postage_date" value="' . esc_attr($prk_woocot_postage_date) . '" /></p>';
        echo '</div>';
        echo '
		<script>
		jQuery(document).ready(function($) {
			$("#prk_woocot_postage_date").persianDatepicker({
			initialValue: false,
			   cellWidth: 32,
				cellHeight: 30,
				fontSize: 14,
			});
		  });
		  </script>
		';
    }

    public function get_shippers()
    {
        $shippers = [
            'post' => ['name' => 'پست'],
            'tipax' => ['name' => 'تیپاکس']
        ];

        return apply_filters('prk_woocot_shippers', $shippers);
    }

    /**
     * Saves order tracking meta.
     *
     * Hooked into `save_post` action hook.
     *
     * @param int $post_ID
     */
    public function save_meta_box($order_ID, $post = null, $update = null)
    {
        $order = wc_get_order($order_ID);

        if (!$order) {
            return;
        }

        remove_action( 'save_post', array($this,'save_meta_box'));

        $new_order_status = !empty(PRKSMSAppClass::prk_opt('prk_order_status_after_save_tracking_post'))
            ? PRKSMSAppClass::prk_opt('prk_order_status_after_save_tracking_post')
            : 'completed';

        if (isset($_POST['prk_woocot_order_picked_up']) && $_POST['prk_woocot_order_picked_up'] === 'on') {

            if (!empty($order) && $order->get_status() !== $new_order_status && $this->check_valid_set_tracking_post_info($order_ID)) {
                $old_order_status = $order->get_status();
                $order = $this->update_meta_box($order);
                $order->set_status($new_order_status);
                if ($this->is_hpos_enabled()) {
                    remove_action('woocommerce_order_status_changed', 'prkSMSWoocommerceOrder');
                    if (get_post_meta($order->get_id(), 'prk_woocot_order_picked_up', true) === 'yes') {
                        prkSMSWoocommerceOrder($order->get_id(), $old_order_status, $new_order_status);
                    }
                }
            }
            elseif ($order->get_status() === $new_order_status) {
                $order = $this->update_meta_box($order);
            } else {
                $order->update_meta_data('prk_woocot_order_picked_up', 'no');
            }

        } else {
            $order->update_meta_data('prk_woocot_order_picked_up', 'no');
        }

        $order->save();
        add_action( 'save_post', array($this,'save_meta_box'),10,3);
    }

    function check_valid_set_tracking_post_info($order_ID)
    {
        if (empty($_POST['prk_woocot_shipper']) || empty($_POST['prk_woocot_number']) || empty($_POST['prk_woocot_transport_type']) || empty($_POST['prk_woocot_postage_date']) || empty($_POST['prk_woocot_order_picked_up']) || !current_user_can('edit_post', $order_ID)) {
            return false;
        }
        return true;
    }

    function update_meta_box($order)
    {
        if (!$this->check_valid_set_tracking_post_info($order->get_id())) {
            return;
        }

        $prk_woocot_shipper = isset($_POST['prk_woocot_shipper']) ? sanitize_text_field($_POST['prk_woocot_shipper']) : '';
        $prk_woocot_number = isset($_POST['prk_woocot_number']) ? sanitize_text_field($_POST['prk_woocot_number']) : '';
        $prk_woocot_transport_type = isset($_POST['prk_woocot_transport_type']) ? sanitize_text_field($_POST['prk_woocot_transport_type']) : '';
        $prk_woocot_postage_date = isset($_POST['prk_woocot_postage_date']) ? sanitize_text_field($_POST['prk_woocot_postage_date']) : '';

        $order->update_meta_data('prk_woocot_transport_type', $prk_woocot_transport_type);
        $order->update_meta_data('prk_woocot_postage_date', $prk_woocot_postage_date);
        $order->update_meta_data('prk_woocot_shipper', $prk_woocot_shipper);
        $order->update_meta_data('prk_woocot_number', $prk_woocot_number);
        $order->update_meta_data('prk_woocot_order_picked_up', 'yes');

        return $order;
    }

    /**
     * Admin notice template
     *
     * Hooked into 'admin_notices' action.
     */
    public function admin_notice()
    {
        if (get_transient('prk_woocot_error')) {
            echo '<div class="updated error notice is-dismissible">';
            echo '<p>' . esc_html__('Invalid / Missing tracking number or tracking shipper', 'woocommerce-order-tracking') . '.</p>';
            echo '</div>';

            delete_transient('prk_woocot_error');
        }
    }

    /**
     * Hides `prk_woocot_*` custom fields.
     *
     * Hooked into `is_protected_meta` filter hook.
     *
     * @param bool $protected
     * @param int $meta_key
     * @return bool
     */
    public function protect_meta($protected, $meta_key)
    {
        if ('prk_woocot_shipper' === $meta_key || 'prk_woocot_number' === $meta_key || 'prk_woocot_transport_type' === $meta_key || 'prk_woocot_postage_date' === $meta_key || 'prk_woocot_order_picked_up' === $meta_key) {
            return true;
        }

        return $protected;
    }

    /**
     * Displays shipping information.
     *
     * Hooked into `woocommerce_view_order` action hook.
     * Hooked into `woocommerce_email_before_order_table` action hook.
     *
     * @param object $order
     */
    public function information_display($order)
    {
        if (is_object($order)) {
            $order_id = $order->id;
        } elseif (is_int($order)) {
            $order_id = $order;
        }

        $shippers = $this->get_shippers();
        $prk_woocot_shipper = get_post_meta($order_id, 'prk_woocot_shipper', true);
        $prk_woocot_number = get_post_meta($order_id, 'prk_woocot_number', true);

        if (!empty($prk_woocot_shipper) && !empty($prk_woocot_number) && array_key_exists($prk_woocot_shipper, $shippers)) {
            $html_name = sprintf(esc_html__('ارسال شده با %s', 'woocommerce-order-tracking'), esc_html($shippers[$prk_woocot_shipper]['name'])) . '<br />';
            $html_code .= sprintf(esc_html__('کد رهگیری پستی: %s', 'woocommerce-order-tracking'), $tracking = esc_html($prk_woocot_number)) . '<br />';
            // $html .= '<a href="' . esc_url( $shippers[ $prk_woocot_shipper ]['url'] . $prk_woocot_number ) . '" rel="nofollow" target="_blank">' . esc_html__( 'رهگیری', 'woocommerce-order-tracking' )  . '</a>';

            echo apply_filters('prk_woocot_information_display', $html_name, $html_code, $order, $prk_woocot_shipper, $prk_woocot_number);
        }
    }
}

/**
 * Main function.
 *
 * Avoids the use of a global..
 *
 * @return object Plugin Instance
 */
function prk_woocommerce_order_tracking()
{
    return PRK_Woocommerce_Order_Tracking::get_instance();
}

prk_woocommerce_order_tracking();
